*-------------------------------------------------------------------------------
/* title : Compute second set of Instrumental Variables excluding US (IV2noUS)
		   for intra- and inter-industry proxies (see equations 8-9 in paper)		   
   source: World Input-Output Tables, 2016 Release (in Stata format)
		   http://www.wiod.org/database/wiots16	 
   input : $data/wiot_stata_Nov16/WIOT`year'_October16_ROW.dta
   output: $temp/Proxies_IV2noUS.dta */
*-------------------------------------------------------------------------------

*===============================================================================
* Set up
*===============================================================================
*--------------------------SET-UP & READ DIRECTORIES----------------------------
cd												// Current directory of code
do "[0.0] directories.do"

*===============================================================================
* Create instruments using Chinese exports and ROW imports
*===============================================================================

* Use CHN exports to non-sample countries as IV for inports of intermediates
qui foreach country in CHN {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		drop if Country=="TOT"
		drop TOT *57 *58 *59 *60 *61
		rename RNr id
		drop if id>56
		rename Country country
		keep if country=="`country'"
		rename Year year
		forvalues v=1/56 {
			gen EXP_CHN`v'=0	
			foreach c in `"AUS"' `"BRA"' `"CAN"' `"IDN"' `"IND"' `"JPN"' `"KOR"' `"MEX"' `"ROW"' `"RUS"' `"TUR"' `"TWN"' {
				replace EXP_CHN`v' = EXP_CHN`v' + v`c'`v'
			}
		}
		rename id industry
		keep country year industry EXP*
		order country year industry 
		sort industry year	
		save "$temp/EXPCHN_TO_NONSAMPLE_`y'.dta", replace
	}
}

* Use non-sample countries' imports from other non-sample countries as IV for 
* exports of intermediates
qui forvalues y=2000/2014 {
	use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
	drop if Country=="TOT"
	drop TOT *57 *58 *59 *60 *61
	rename RNr id
	drop if id>56
	rename Country country
	rename Year year
	gen tokeep=0
	foreach country in `"AUS"' `"BRA"' `"CAN"' `"IDN"' `"IND"' `"JPN"' `"KOR"' `"MEX"' `"ROW"' `"RUS"' `"TUR"' `"TWN"' {
		replace tokeep=1 if country=="`country'" 				// no imports from sample countries
		forvalues v=1/56 {
			replace v`country'`v' = 0 if country=="`country'" 	// only imports, no domestic intermediates
		}
	}
	keep if tokeep==1
	drop tokeep
	forvalues v=1/56 {
		gen vINST`v'=0
		foreach country in `"CHN"' {
			replace vINST`v' = vINST`v' + v`country'`v'
		}
	}
	gcollapse (sum) vINST* , by(id year)
	gen country="INST"
	order country
	rename id industry
	save "$temp/IMP_NONSAMPLE_`y'.dta", replace
}

*===============================================================================
* Inter-industry proxies that include all supply links (broad)
*===============================================================================

* STEP 1a: construct "matrices" of weights from domestic IO tables
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$data/wiot_stata_Nov16/WIOT`y'_October16_ROW.dta", clear
		keep if Country=="`country'"
		rename Country country
		rename RNr id
		drop if id>56
		drop *57 *58 *59 *60 *61
		keep country id *`country'*
		rename v`country'* i*

		gegen i_rowsum=rowtotal(i1-i56)
		forvalues v=1/56 {
			gen c_down`v' = i`v'/i_rowsum
			replace c_down`v'=0 if c_down`v'==. 	// need zeros for matrix manipulation later on
		}
		forvalues v=1/56 {
			gegen i`v'_colsum=total(i`v')
			gen c_up`v' = i`v'/i`v'_colsum
			replace c_up`v'=0 if c_up`v'==. 		// need zeros for matrix manipulation later on
			drop i`v'_colsum
		}
		forvalues v=1/56 {
			gen c_down`v'_0D = c_down`v' 
			replace c_down`v'_0D = 0 if id==`v' 
		}
		forvalues v=1/56 {
			gen c_up`v'_0D = c_up`v' 
			replace c_up`v'_0D = 0 if id==`v' 
		}
		rename id bid
		drop i_rowsum i*
		gen year = `y'
		rename bid industry
		order country year industry
		compress
		save "$temp/COEF`country'_`y'.dta", replace
	}
	noi dis "`country'"
}

* STEP 1b: IV for overall IM in other industries --> weighted chinese exports
* 		   using share of inputs for upstream industries to weigh total chinese 
*		   exports by given industry; "matrices" of weights from above
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$temp/COEF`country'_`y'.dta", clear
		cap drop country
		merge 1:1 industry year using "$temp/EXPCHN_TO_NONSAMPLE_`y'.dta"
		drop _merge
		cap drop country
		gen country = "`country'"

		gen EXP_CHN=0	
		forvalues v=1/56 {
			replace EXP_CHN = EXP_CHN + EXP_CHN`v'
			drop EXP_CHN`v'
		}

		mkmat EXP_CHN, matrix(EXPCHN)
		mkmat c_up1-c_up56, matrix(UP)
		mkmat c_up1_0D-c_up56_0D, matrix(UPZD)

		foreach type in EXPCHN {
			mat upp0d`type' = UPZD'*`type'
			svmat upp0d`type', names(up0d`type')
			rename up0d`type'* up0d`type'
			mat upp`type' = UP'*`type'
			svmat upp`type', names(up`type')
			rename up`type'* up`type'
		}
		keep country industry year up*
		compress
		save "$temp/UPEXPCHN_`country'_`y'.dta", replace
	}
	noi dis "`country'"
}

* STEP 1c: IV for overall EX in other industries --> weighted ROW imports using
* 		   share of output for downstream industries as weights
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$temp/IMP_NONSAMPLE_`y'.dta", replace
		gcollapse (sum) vINST* , by(year)
		greshape long vINST, i(year) j(industry)
		rename vINST IMP
		cap drop country
		merge 1:1 industry year using "$temp/COEF`country'_`y'.dta"
		drop _merge
		cap drop country
		gen country = "`country'"

		mkmat IMP, matrix(IMP)
		mkmat c_down1-c_down56, matrix(DOWN)
		mkmat c_down1_0D-c_down56_0D, matrix(DOWNZD)

		foreach type in IMP {
			mat dwn0d`type' = DOWNZD*`type'
			svmat dwn0d`type', names(down0d`type')
			rename down0d`type'* down0d`type'
			mat dwn`type' = DOWN*`type'
			svmat dwn`type', names(down`type')
			rename down`type'* down`type'
		}
		keep country industry year down*
		compress
		save "$temp/DOWNIMP_`country'_`y'.dta", replace
	}
	noi dis "`country'"
}

* STEP 2: IVs for downIM, upEX, upIM, downEX with time-varying weights
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		use "$temp/COEF`country'_`y'.dta", clear
		cap drop country
		merge 1:1 industry year using "$temp/UPEXPCHN_`country'_`y'.dta", keepusing(up0dEXPCHN upEXPCHN)
		drop _merge
		cap drop country
		merge 1:1 industry year using "$temp/DOWNIMP_`country'_`y'.dta", keepusing(down0dIMP downIMP)
		drop _merge
		cap drop country
		merge 1:1 industry year using "$temp/IMP_NONSAMPLE_`y'.dta", keepusing(vINST1-vINST56)
		drop _merge
		cap drop country
		merge 1:1 industry year using "$temp/EXPCHN_TO_NONSAMPLE_`y'.dta", keepusing(EXP_CHN1-EXP_CHN56)
		drop _merge
		cap drop country
		gen country="`country'"

		order country industry year 
		
		mkmat c_down1-c_down56, matrix(DOWN)
		mkmat c_up1-c_up56, matrix(UP)
		mkmat c_down1_0D-c_down56_0D, matrix(DOWNZD)
		mkmat c_up1_0D-c_up56_0D, matrix(UPZD)

		mkmat up0dEXPCHN, matrix(UP0CN)
		mkmat upEXPCHN, matrix(UPCN) 
		mkmat down0dIMP, matrix(DW0IM)
		mkmat downIMP, matrix(DWIM)
		mkmat vINST1-vINST56, matrix(IMP)
		mkmat EXP_CHN1-EXP_CHN56, matrix(EXP)

		foreach type in UP0CN { //up0dIM_iv2!!
			mat upp0d`type' = UPZD'*`type'
			svmat upp0d`type', names(up0d`type')
			rename up0d`type'* up0dIM_iv2
		}
		foreach type in UPCN { //upIM_iv2!!
			mat upp`type' = UP'*`type'
			svmat upp`type', names(up`type')
			rename up`type'* upIM_iv2
		}
		foreach type in DW0IM { //down0dEX_iv2
			mat dwn0d`type' = DOWNZD*`type'
			svmat dwn0d`type', names(down0d`type')
			rename down0d`type'* down0dEX_iv2
		}
		foreach type in DWIM { //downEX_iv2
			mat dwn`type' = DOWN*`type'
			svmat dwn`type', names(down`type')
			rename down`type'* downEX_iv2
		}
		foreach type in EXP { //downIM!
			mat dwn0d`type' = DOWNZD*`type'
			svmat dwn0d`type', names(down0dIM)
			mat dwn`type' = DOWN*`type'
			svmat dwn`type', names(downIM)
			}
		foreach type in IMP { //upEX!!
			mat upp0d`type' = UPZD'*`type'
			svmat upp0d`type', names(up0dEX)
			mat upp`type' = UP'*`type'
			svmat upp`type', names(upEX)
		}
		foreach var in down0dIM downIM up0dEX upEX {
			gen `var'_iv2 = .
		}
		
		forvalues v=1/56 {
			foreach var in down0dIM downIM up0dEX upEX {
				replace `var'_iv2 = `var'`v' if industry==`v' //need diagonal elements
				drop `var'`v' 
			}
		}
		keep country industry year *_iv2 up0dEXPCHN upEXPCHN down0dIMP downIMP
		rename up0dEXPCHN IM_iv2_0d
		rename upEXPCHN IM_iv2
		rename down0dIMP EX_iv2_0d
		rename downIMP EX_iv2
		save "$temp/IM`country'_`y'_TV_IV2noUS.dta", replace
	}
	noi dis "`country'"
}

* Append data
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	use "$temp/IM`country'_2000_TV_IV2noUS.dta", replace
	forvalues y=2001/2014 {
		append using "$temp/IM`country'_`y'_TV_IV2noUS.dta"
	}
	sort country industry year
	save "$temp/Proxies_`country'_IV2noUS.dta", replace
}

use "$temp/Proxies_AUT_IV2noUS.dta", clear
qui foreach country in BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	append using "$temp/Proxies_`country'_IV2noUS.dta"
}
sort country industry year
order country industry year IM* EX* down0dIM_iv2 up0dIM_iv2 up0dEX_iv2 down0dEX_iv2 downIM_iv2 upIM_iv2 upEX_iv2 downEX_iv2
rename *iv2 *iv2noUS
rename *iv2_0d *iv2_0dnoUS

save "$temp/Proxies_IV2noUS.dta", replace

* Housekeeping
qui foreach country in AUT BEL DEU DNK ESP FIN FRA GBR GRC IRL ITA NLD NOR PRT SWE BGR CZE EST HRV HUN LTU LVA POL ROU SVK SVN {
	forvalues y=2000/2014 {
		cap erase "$temp/UPEXPCHN_`country'_`y'.dta"
		cap erase "$temp/DOWNIMP_`country'_`y'.dta"
		cap erase "$temp/IMP_NONSAMPLE_`y'.dta"
		cap erase "$temp/EXPCHN_TO_NONSAMPLE_`y'.dta"
		cap erase "$temp/IM`country'_`y'_IV2noUS.dta"
		cap erase "$temp/IM`country'_`y'_TV_IV2noUS.dta"
		cap erase "$temp/IM`country'_`y'_00_IV2noUS.dta"
		cap erase "$temp/COEF`country'_`y'.dta"
		cap erase "$temp/COEF`country'_`y'_IV2noUS.dta"
		cap erase "$temp/EX`country'_`y'_IV2noUS.dta"
		cap erase "$temp/Proxies_`country'_IV2noUS.dta"
		cap erase "$temp/Proxies_`country'_00_IV2noUS.dta"
		cap erase "$temp/Proxies_00_IV2noUS.dta"
		cap erase "$temp/IMP_`y'.dta"
		cap erase "$temp/IMPNONSAMPLE_`y'.dta"
		cap erase "$temp/EXPCHN_`y'_IVCN.dta"
	}
}
